﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lambda/LambdaRequest.h>
#include <aws/lambda/Lambda_EXPORTS.h>
#include <aws/lambda/model/AliasRoutingConfiguration.h>

#include <utility>

namespace Aws {
namespace Lambda {
namespace Model {

/**
 */
class CreateAliasRequest : public LambdaRequest {
 public:
  AWS_LAMBDA_API CreateAliasRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAlias"; }

  AWS_LAMBDA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name or ARN of the Lambda function.</p> <p class="title"> <b>Name
   * formats</b> </p> <ul> <li> <p> <b>Function name</b> -
   * <code>MyFunction</code>.</p> </li> <li> <p> <b>Function ARN</b> -
   * <code>arn:aws:lambda:us-west-2:123456789012:function:MyFunction</code>.</p>
   * </li> <li> <p> <b>Partial ARN</b> -
   * <code>123456789012:function:MyFunction</code>.</p> </li> </ul> <p>The length
   * constraint applies only to the full ARN. If you specify only the function name,
   * it is limited to 64 characters in length.</p>
   */
  inline const Aws::String& GetFunctionName() const { return m_functionName; }
  inline bool FunctionNameHasBeenSet() const { return m_functionNameHasBeenSet; }
  template <typename FunctionNameT = Aws::String>
  void SetFunctionName(FunctionNameT&& value) {
    m_functionNameHasBeenSet = true;
    m_functionName = std::forward<FunctionNameT>(value);
  }
  template <typename FunctionNameT = Aws::String>
  CreateAliasRequest& WithFunctionName(FunctionNameT&& value) {
    SetFunctionName(std::forward<FunctionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the alias.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateAliasRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The function version that the alias invokes.</p>
   */
  inline const Aws::String& GetFunctionVersion() const { return m_functionVersion; }
  inline bool FunctionVersionHasBeenSet() const { return m_functionVersionHasBeenSet; }
  template <typename FunctionVersionT = Aws::String>
  void SetFunctionVersion(FunctionVersionT&& value) {
    m_functionVersionHasBeenSet = true;
    m_functionVersion = std::forward<FunctionVersionT>(value);
  }
  template <typename FunctionVersionT = Aws::String>
  CreateAliasRequest& WithFunctionVersion(FunctionVersionT&& value) {
    SetFunctionVersion(std::forward<FunctionVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the alias.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateAliasRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-aliases.html#configuring-alias-routing">routing
   * configuration</a> of the alias.</p>
   */
  inline const AliasRoutingConfiguration& GetRoutingConfig() const { return m_routingConfig; }
  inline bool RoutingConfigHasBeenSet() const { return m_routingConfigHasBeenSet; }
  template <typename RoutingConfigT = AliasRoutingConfiguration>
  void SetRoutingConfig(RoutingConfigT&& value) {
    m_routingConfigHasBeenSet = true;
    m_routingConfig = std::forward<RoutingConfigT>(value);
  }
  template <typename RoutingConfigT = AliasRoutingConfiguration>
  CreateAliasRequest& WithRoutingConfig(RoutingConfigT&& value) {
    SetRoutingConfig(std::forward<RoutingConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_functionName;
  bool m_functionNameHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_functionVersion;
  bool m_functionVersionHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  AliasRoutingConfiguration m_routingConfig;
  bool m_routingConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lambda
}  // namespace Aws
