﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iottwinmaker/IoTTwinMakerRequest.h>
#include <aws/iottwinmaker/IoTTwinMaker_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IoTTwinMaker {
namespace Model {

/**
 */
class CreateSyncJobRequest : public IoTTwinMakerRequest {
 public:
  AWS_IOTTWINMAKER_API CreateSyncJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateSyncJob"; }

  AWS_IOTTWINMAKER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The workspace ID.</p>
   */
  inline const Aws::String& GetWorkspaceId() const { return m_workspaceId; }
  inline bool WorkspaceIdHasBeenSet() const { return m_workspaceIdHasBeenSet; }
  template <typename WorkspaceIdT = Aws::String>
  void SetWorkspaceId(WorkspaceIdT&& value) {
    m_workspaceIdHasBeenSet = true;
    m_workspaceId = std::forward<WorkspaceIdT>(value);
  }
  template <typename WorkspaceIdT = Aws::String>
  CreateSyncJobRequest& WithWorkspaceId(WorkspaceIdT&& value) {
    SetWorkspaceId(std::forward<WorkspaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sync source.</p>  <p>Currently the only supported syncSoource is
   * <code>SITEWISE </code>.</p>
   */
  inline const Aws::String& GetSyncSource() const { return m_syncSource; }
  inline bool SyncSourceHasBeenSet() const { return m_syncSourceHasBeenSet; }
  template <typename SyncSourceT = Aws::String>
  void SetSyncSource(SyncSourceT&& value) {
    m_syncSourceHasBeenSet = true;
    m_syncSource = std::forward<SyncSourceT>(value);
  }
  template <typename SyncSourceT = Aws::String>
  CreateSyncJobRequest& WithSyncSource(SyncSourceT&& value) {
    SetSyncSource(std::forward<SyncSourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The SyncJob IAM role. This IAM role is used by the SyncJob to read from the
   * syncSource, and create, update, or delete the corresponding resources.</p>
   */
  inline const Aws::String& GetSyncRole() const { return m_syncRole; }
  inline bool SyncRoleHasBeenSet() const { return m_syncRoleHasBeenSet; }
  template <typename SyncRoleT = Aws::String>
  void SetSyncRole(SyncRoleT&& value) {
    m_syncRoleHasBeenSet = true;
    m_syncRole = std::forward<SyncRoleT>(value);
  }
  template <typename SyncRoleT = Aws::String>
  CreateSyncJobRequest& WithSyncRole(SyncRoleT&& value) {
    SetSyncRole(std::forward<SyncRoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The SyncJob tags.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateSyncJobRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateSyncJobRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_workspaceId;

  Aws::String m_syncSource;

  Aws::String m_syncRole;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_workspaceIdHasBeenSet = false;
  bool m_syncSourceHasBeenSet = false;
  bool m_syncRoleHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTTwinMaker
}  // namespace Aws
