/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.encryptionsdk.caching;

import com.amazonaws.encryptionsdk.CryptoAlgorithm;
import com.amazonaws.encryptionsdk.CryptoMaterialsManager;
import com.amazonaws.encryptionsdk.DefaultCryptoMaterialsManager;
import com.amazonaws.encryptionsdk.MasterKeyProvider;
import com.amazonaws.encryptionsdk.caching.CryptoMaterialsCache;
import com.amazonaws.encryptionsdk.exception.AwsCryptoException;
import com.amazonaws.encryptionsdk.internal.EncryptionContextSerializer;
import com.amazonaws.encryptionsdk.internal.Utils;
import com.amazonaws.encryptionsdk.model.DecryptionMaterials;
import com.amazonaws.encryptionsdk.model.DecryptionMaterialsRequest;
import com.amazonaws.encryptionsdk.model.EncryptionMaterials;
import com.amazonaws.encryptionsdk.model.EncryptionMaterialsRequest;
import com.amazonaws.encryptionsdk.model.KeyBlob;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

public class CachingCryptoMaterialsManager
implements CryptoMaterialsManager {
    private static final String CACHE_ID_HASH_ALGORITHM = "SHA-512";
    private static final long MAX_MESSAGE_USE_LIMIT = 0x100000000L;
    private static final long MAX_BYTE_USE_LIMIT = Long.MAX_VALUE;
    private final CryptoMaterialsManager backingCMM;
    private final CryptoMaterialsCache cache;
    private final byte[] partitionIdHash;
    private final String partitionId;
    private final long maxAgeMs;
    private final long messageUseLimit;
    private final long byteUseLimit;
    private final CryptoMaterialsCache.CacheHint hint = new CryptoMaterialsCache.CacheHint(){

        @Override
        public long getMaxAgeMillis() {
            return CachingCryptoMaterialsManager.this.maxAgeMs;
        }
    };

    public static Builder newBuilder() {
        return new Builder();
    }

    private CachingCryptoMaterialsManager(Builder builder) {
        this.backingCMM = builder.backingCMM;
        this.cache = builder.cache;
        this.partitionId = builder.partitionId != null ? builder.partitionId : UUID.randomUUID().toString();
        this.maxAgeMs = builder.maxAge;
        this.messageUseLimit = builder.messageUseLimit;
        this.byteUseLimit = builder.byteUseLimit;
        try {
            this.partitionIdHash = MessageDigest.getInstance(CACHE_ID_HASH_ALGORITHM).digest(this.partitionId.getBytes(StandardCharsets.UTF_8));
        }
        catch (GeneralSecurityException e) {
            throw new AwsCryptoException(e);
        }
    }

    @Override
    public EncryptionMaterials getMaterialsForEncrypt(EncryptionMaterialsRequest request) {
        EncryptionMaterials result;
        if (request.getPlaintextSize() == -1L) {
            return this.backingCMM.getMaterialsForEncrypt(request);
        }
        EncryptionMaterialsRequest upstreamRequest = request.toBuilder().setPlaintext(null).setPlaintextSize(-1L).build();
        byte[] cacheId = this.getCacheIdentifier(upstreamRequest);
        CryptoMaterialsCache.UsageStats increment = this.initialIncrementForRequest(request);
        if (increment.getBytesEncrypted() >= this.byteUseLimit) {
            return this.backingCMM.getMaterialsForEncrypt(request);
        }
        CryptoMaterialsCache.EncryptCacheEntry entry = this.cache.getEntryForEncrypt(cacheId, increment);
        if (entry != null && !this.isEntryExpired(entry.getEntryCreationTime()) && !this.hasExceededLimits(entry.getUsageStats())) {
            return entry.getResult();
        }
        if (entry != null) {
            entry.invalidate();
        }
        if ((result = this.backingCMM.getMaterialsForEncrypt(request)).getAlgorithm().isSafeToCache()) {
            this.cache.putEntryForEncrypt(cacheId, result, this.hint, this.initialIncrementForRequest(request));
        }
        return result;
    }

    private boolean hasExceededLimits(CryptoMaterialsCache.UsageStats stats) {
        return stats.getBytesEncrypted() > this.byteUseLimit || stats.getMessagesEncrypted() > this.messageUseLimit;
    }

    private boolean isEntryExpired(long entryCreationTime) {
        return System.currentTimeMillis() - entryCreationTime > this.maxAgeMs;
    }

    private CryptoMaterialsCache.UsageStats initialIncrementForRequest(EncryptionMaterialsRequest request) {
        return new CryptoMaterialsCache.UsageStats(request.getPlaintextSize(), 1L);
    }

    @Override
    public DecryptionMaterials decryptMaterials(DecryptionMaterialsRequest request) {
        byte[] cacheId = this.getCacheIdentifier(request);
        CryptoMaterialsCache.DecryptCacheEntry entry = this.cache.getEntryForDecrypt(cacheId);
        if (entry != null && !this.isEntryExpired(entry.getEntryCreationTime())) {
            return entry.getResult();
        }
        DecryptionMaterials result = this.backingCMM.decryptMaterials(request);
        this.cache.putEntryForDecrypt(cacheId, result, this.hint);
        return result;
    }

    private byte[] getCacheIdentifier(EncryptionMaterialsRequest req) {
        try {
            MessageDigest digest = MessageDigest.getInstance(CACHE_ID_HASH_ALGORITHM);
            digest.update(this.partitionIdHash);
            CryptoAlgorithm algorithm = req.getRequestedAlgorithm();
            digest.update((byte)(algorithm != null ? 1 : 0));
            if (algorithm != null) {
                this.updateDigestWithAlgorithm(digest, algorithm);
            }
            digest.update(MessageDigest.getInstance(CACHE_ID_HASH_ALGORITHM).digest(EncryptionContextSerializer.serialize(req.getContext())));
            return digest.digest();
        }
        catch (GeneralSecurityException e) {
            throw new AwsCryptoException(e);
        }
    }

    private byte[] getCacheIdentifier(DecryptionMaterialsRequest req) {
        try {
            MessageDigest digest = MessageDigest.getInstance(CACHE_ID_HASH_ALGORITHM);
            byte[] hashOfContext = digest.digest(EncryptionContextSerializer.serialize(req.getEncryptionContext()));
            ArrayList<byte[]> keyBlobHashes = new ArrayList<byte[]>(req.getEncryptedDataKeys().size());
            for (KeyBlob blob : req.getEncryptedDataKeys()) {
                keyBlobHashes.add(digest.digest(blob.toByteArray()));
            }
            keyBlobHashes.sort(new Utils.ComparingByteArrays());
            digest.update(this.partitionIdHash);
            this.updateDigestWithAlgorithm(digest, req.getAlgorithm());
            keyBlobHashes.forEach(digest::update);
            digest.update(new byte[digest.getDigestLength()]);
            digest.update(hashOfContext);
            return digest.digest();
        }
        catch (GeneralSecurityException e) {
            throw new AwsCryptoException(e);
        }
    }

    private void updateDigestWithAlgorithm(MessageDigest digest, CryptoAlgorithm algorithm) {
        short algId = algorithm.getValue();
        digest.update(new byte[]{(byte)(algId >> 8), (byte)algId});
    }

    public static class Builder {
        private CryptoMaterialsManager backingCMM;
        private CryptoMaterialsCache cache;
        private String partitionId = null;
        private long maxAge = 0L;
        private long messageUseLimit = 0x100000000L;
        private long byteUseLimit = Long.MAX_VALUE;

        private Builder() {
        }

        public Builder withBackingMaterialsManager(CryptoMaterialsManager backingCMM) {
            this.backingCMM = backingCMM;
            return this;
        }

        public Builder withMasterKeyProvider(MasterKeyProvider mkp) {
            return this.withBackingMaterialsManager(new DefaultCryptoMaterialsManager(mkp));
        }

        public Builder withCache(CryptoMaterialsCache cache) {
            this.cache = cache;
            return this;
        }

        public Builder withPartitionId(String partitionId) {
            this.partitionId = partitionId;
            return this;
        }

        public Builder withMaxAge(long maxAge, TimeUnit units) {
            if (maxAge <= 0L) {
                throw new IllegalArgumentException("Max age must be positive");
            }
            this.maxAge = units.toMillis(maxAge);
            return this;
        }

        public Builder withMessageUseLimit(long messageUseLimit) {
            if (messageUseLimit <= 0L) {
                throw new IllegalArgumentException("Message use limit must be positive");
            }
            if (messageUseLimit > 0x100000000L) {
                throw new IllegalArgumentException("Message use limit exceeds limit of 4294967296");
            }
            this.messageUseLimit = messageUseLimit;
            return this;
        }

        public Builder withByteUseLimit(long byteUseLimit) {
            if (byteUseLimit < 0L) {
                throw new IllegalArgumentException("Byte use limit must be non-negative");
            }
            if (byteUseLimit > Long.MAX_VALUE) {
                throw new IllegalArgumentException("Byte use limit exceeds maximum of 9223372036854775807");
            }
            this.byteUseLimit = byteUseLimit;
            return this;
        }

        public CachingCryptoMaterialsManager build() {
            if (this.backingCMM == null) {
                throw new IllegalArgumentException("Backing CMM must be set");
            }
            if (this.cache == null) {
                throw new IllegalArgumentException("Cache must be set");
            }
            if (this.maxAge <= 0L) {
                throw new IllegalArgumentException("Max age must be set");
            }
            return new CachingCryptoMaterialsManager(this);
        }
    }
}

